#! /usr/local/bin/python
import numpy
import pylab

import scipy.stats
from scipy.interpolate import interp1d

###  load filtered, 10my binned data
carb_filt=numpy.ones(shape=(len(numpy.load("carb_10my_t.npy")),2))
carb_filt[:,0]=numpy.load("carb_10my_t.npy") #time values for carbonates
carb_filt[:,1]=numpy.load("carb_10my_v.npy") #isotope values for carbonates
org_filt=numpy.ones(shape=(len(numpy.load("org_10my_t.npy")),2))
org_filt[:,0]=numpy.load("org_10my_t.npy") #time values for organics
org_filt[:,1]=numpy.load("org_10my_v.npy") #isotope values for organics

#import stats module for LOWESS calculation
import statsmodels.api as sm

#loop over both organics and carbonates
for jj in range(0,2):
    
    pylab.figure() #create figure
    
    if jj==0: #organics
        data=org_filt  
        fo=0.1231 #optimal window size for organics (see main text for leave-out-one method of calculation)
        pylab.title("organic") #figure title
    else:
        data=carb_filt  
        fo=0.1449 #optimal window size for carbonates (see main text for leave-out-one method of calculation)
        pylab.title("carbonate") #figure title
    
    N=1000 #number of iterations to generate uncertainties 
    #(result virtually identical for 10000 iterations)
    
    #use built in lowess function
    of1=sm.nonparametric.lowess(data[:,1], data[:,0], frac=fo, it=3, delta=0.0, is_sorted=False, missing='drop', return_sorted=True)
    residuals=data[:,1]-of1[:,1] #residuals from LOWESS fit
    
    #define master_array: will contain resampled smoothed curves, and will be used to generate uncertainties
    master_array=numpy.ones(shape=(N,len(data[:,1])))
    
    #plot LOWESS result with original data
    pylab.plot(data[:,0],data[:,1],'gx') #data as green crosses
    pylab.plot(of1[:,0],of1[:,1],'k-') #smoothed LOWESS curve in black
    
    #now find uncertainty in smoothed result
    for i in range(0,N): #number of iterations, N, defined above
        
        print i,"of",N #print progress
        
        #convert both time and isotope vectors to single array, temp_data
        temp_data=numpy.ones(shape=(len(data[:,0]),2))
        time=data[:,0]
        val=data[:,1]
        temp_data[:,0]=time
        temp_data[:,1]=val
                
        ###### resample residuals with replacement then reapply smoothing algorithm:
        re_resid=numpy.random.choice(residuals, size=len(residuals), replace=True)  #resample residuals
        re_resid=re_resid-numpy.mean(re_resid) #centre resampled residuals on zero
        new_data=re_resid+temp_data[:,1] #add these resampled, centred residuals back to the original smoothed curve
        of=sm.nonparametric.lowess(new_data, temp_data[:,0], frac=fo, it=3, delta=0.0, is_sorted=False, missing='drop', return_sorted=True) #redo smoothing
        
        #take the output and add it to the array containing all iterations (this array will be used to generate uncertainties)
        master_array[i,:]=of[:,1]
    
    #calculate 95% confidence interval for smoothed curve
    confidence=scipy.stats.scoreatpercentile(master_array,[2.5,50,97.5], interpolation_method='fraction',axis=0)  
    #plot 95% confidence interval, and median curve
    pylab.plot(data[:,0],confidence[0],'r--') #plots lower bound of 95% confidence
    pylab.plot(data[:,0],confidence[1],'r') #plots smoothed curve
    pylab.plot(data[:,0],confidence[2],'r--') #plots upper bound of 95% confidence

    #label figures
    if jj==0:  #organic labels
        organic_array=master_array #This creates an array with all the organic realizations, for use in forg calculations below
        pylab.title("Organic LOWESS") #figure title
        pylab.xlabel("time (mya)") #x axis label
        pylab.ylabel("delta 13C") #y axis label
    if jj==1:
        carb_array=master_array
        pylab.title("Carbonate LOWESS") #figure title
        pylab.xlabel("time (mya)") #x axis label
        pylab.ylabel("delta 13C") #y axis label
        
#define median curves for use below
confidence_o=scipy.stats.scoreatpercentile(organic_array,[50], interpolation_method='fraction',axis=0) #median organic curve
confidence_c=scipy.stats.scoreatpercentile(carb_array,[50], interpolation_method='fraction',axis=0) #median carbonate curve

#load interpolation function
from scipy.interpolate import interp1d

#in this section the smoothed organic and carbonate curves (with uncertainties) 
#are used to calcuate smoothed forg (with uncertainty)

t=numpy.linspace(5,3600,1000) #define new time axis for calculating forg(t), from 5 Ma to 3600 Ma in 1000 steps
#create interpolated functions for organic and carbonate time series.
#we need to interpolate because organics and carbonates have different time vectors
org_fun = interp1d(org_filt[:,0],confidence_o[0]) #interpolation function for organics
carb_fun = interp1d(carb_filt[:,0],confidence_c[0]) #interpolation function for carbonates
new_org=org_fun(t) #organics(t) using common time vector defined above
new_carb=carb_fun(t) #carbonates(t) using common time vector defined above
#apply interpolated functions to get forg time series
f=(-5-new_carb)/(new_org-new_carb)


pylab.figure() #create new figure for forg results

#now calculate uncertainty in smoothed forg curve by applying interpolation to each resampled curve
#and then adding forg realization to the new array, mega_f:
mega_f=numpy.ones(shape=(N,len(t)))

for jj in range(0,N): #loop over all realizations of organic and carbonate curves
    org_fun = interp1d(org_filt[:,0],organic_array[jj,:]) #create interpolation function for jj-th organic realization
    carb_fun = interp1d(carb_filt[:,0],carb_array[jj,:]) #create interpolation function for jj-th carbonate realization
    new_org=org_fun(t) #jj-th interpolated organic curve
    new_carb=carb_fun(t) #jj-th interpolated carbonate curve 
    f=(-5-new_carb)/(new_org-new_carb) #smoothed forg from the jj-th iteration
    mega_f[jj,:]=f #add this realization of forg to the array

#calculate 95% confidence intervals on forg as a function of time
confidence_f=scipy.stats.scoreatpercentile(mega_f,[2.5,50,97.5], interpolation_method='fraction',axis=0)
#plot forg with confidence interval
pylab.plot(t,confidence_f[0],'r--') #lower bound of 95% confidence on forg
pylab.plot(t,confidence_f[1],'r') #median smoothed forg
pylab.plot(t,confidence_f[2],'r--') #upper bound of 95% confidence on forg
pylab.title("forg LOWESS") #figure title
pylab.xlabel("time (mya)") #x xais label

## FOR OUTPUTTING:
# save arrays for use in plot_LOWESS_Kernel
numpy.save("mega_f_LOWESS",mega_f)
numpy.save("org_array_LOWESS",organic_array)
numpy.save("carb_array_LOWESS",carb_array)

#show all the functions plotted above
pylab.show()   

import sys
sys.exit()
