#! /usr/local/bin/python
import numpy
from classical_GLS import GLS #uses classical GLS regression

########################################
#The shuffle function performs bootstrap GLS regression
#See appendix A.2 and main text for detailed theory.

# This function takes as inputs:
# t: time vector in correct matrix format
# r: scaled residuals from classical GLS fit
# B1: best fit gradient from classical GLS fit
# B0: best fit intercept from classical GLS fit
# Sp: best fit error from classical GLS fit
# Tau: best fit time constant fom classical GLS fit

# The function returns the best fit gradient and intercept with errors from bootstrap GLS
########################################

def shuffle(t,r,B1,B0,Sp,Tau):

    n=numpy.size(t[:,1]) #number of data points
    tau_est=Tau #Tau value used for fit
    
    ahat_prime=numpy.ones([n,1])
    ahat_prime[0,0]=0
    for i in range(1,n):
        ahat_prime[i,0]=numpy.exp(-(t[i,1]-t[i-1,1])/tau_est) #equation A.6
    
    
    e=numpy.ones([n,1]) #uncorrelated residuals
    e[0,0]=r[0,0]
    for i in range(1,n):
        e[i,0]=(r[i,0]-ahat_prime[i,0]*r[i-1,0])*(1-ahat_prime[i,0]**2)**-0.5 #equation A.7
    
    e_mean=e-numpy.sum(e[1:,0]/(n-1.0)) #centred uncorrelated residuals, equation A.8
        
    V=numpy.ones([n,n]) #covariance matrix
    for i in range(0,n):
        for k in range(0,n):
            V[i,k]=Sp[i,0]*Sp[k,0]*numpy.exp(-abs(t[i,1]-t[k,1])/tau_est)
            
    for i in range(0,1):
        
        #sample uncorrelated residuals with replacement
        estar=numpy.random.choice(e_mean[:,0], size=len(e_mean), replace=True) # equation A.9
        
        rstar=numpy.ones([n,1])
        rstar[0,0]=estar[0]
        
        for i in range(1,n): #resampled correlated residuals equation A.10
            rstar[i,0]=ahat_prime[i,0]*rstar[i-1,0]+estar[i]*(1-ahat_prime[i,0]**2)**0.5

        # equation A.11, add trend back to resampled residuals
        xstar=B0+B1*t+rstar*Sp
        xst=numpy.ones([len(xstar),1])
        xst[:,0]=xstar[:,1]
        
        # fit new resampled data, step 7.
        (B11,B00,er1,er2,SSQG)=GLS(t,xst,Sp,V,Tau) 
        
    return B11,B00,er1,er2 #return new fitted values (gradient, intercept, error in gradient, error in intercept, respectively)
        

